import os
import requests
from io import BytesIO

# HTTP
from django.shortcuts import render,HttpResponse
from http import HTTPStatus
from django.http import JsonResponse
from django.views.decorators.csrf import csrf_exempt

# JSON
import json
# DATETIME
from datetime import datetime,date

import locale

# Utilidades de funciones
from functools import partial


from clientes.models import Clientes
from oficinas.models import Oficinas
# Utilidades de archivos
from utils.file_utils import upload_image_to_server, upload_file_to_server
from utils.static_constants import STATIC_PREFIX_PATH, STATIC_FULL_URL_PATH

# Modelos
from .models import ReporteGardclean, DatosGeneralesReporteG, SeccionReporteGardclean, ImagenEvidenciaRg, \
    ProyectoGardclean, ZonaReporteGc, SeccionProyectoGc, ClienteZonaReporteGc

from django.db.models import Q

# PDF
from reportlab.lib.styles import ParagraphStyle, getSampleStyleSheet
from reportlab.lib.units import cm, inch, mm
from reportlab.pdfbase import pdfmetrics
from reportlab.pdfbase.ttfonts import TTFont
from reportlab.pdfgen import canvas
from reportlab.platypus import (
    Image,
    Paragraph,
    SimpleDocTemplate,
    Spacer,
    Table,
    TableStyle,
    XPreformatted,
    PageBreak,
    Flowable
)
from reportlab.graphics.shapes import Drawing, Line
from reportlab.lib import colors
from reportlab.lib.pagesizes import landscape, letter, portrait, A4

from reportlab.lib.enums import TA_CENTER
from reportlab.platypus.flowables import KeepTogether
from reportlab.lib import utils

from reportlab.lib.utils import ImageReader

# Pillow
from PIL import Image as PILImage


# Static maps
from api.views import get_static_map_by_address_local

# URL encode
import urllib.parse


def test(request):
    return HttpResponse("Test")

# ** Vistas **

# ** Metodos de utilidad **
def get_reportlab_image(img_path,width_percent,height):
    img = utils.ImageReader(img_path)
    iw, ih = img.getSize()
    aspect = ih / float(iw)

    # Calcular ancho basado en el alto fijo para no deformar
    # Si el ancho calculado supera el maximo, forzamos el ancho maximo
    ancho_calculado = height / aspect

    # Calcula el ancho objetivo
    target_width = A4[0] * (width_percent / 100.0)

    #return Image(img_path,width=target_width,height=(target_width * aspect))
    return Image(img_path,width=target_width,height=height)

def get_static_map_img_cropped(api_url, target_width, target_height):
    # 1. Descargar la imagen de Google Maps
    response = requests.get(api_url)
    if response.status_code != 200:
        return None # O manejar el error

    # 2. Abrirla en Pillow usando BytesIO (en memoria)
    img = PILImage.open(BytesIO(response.content))
    img_width, img_height = img.size

    # 3. Lógica de recorte (Crop) para mantener Proporción
    aspect_target = target_width / target_height
    aspect_img = img_width / img_height

    if aspect_img > aspect_target:
        new_width = img_height * aspect_target
        left = (img_width - new_width) / 2
        img = img.crop((left, 0, left + new_width, img_height))
    else:
        new_height = img_width / aspect_target
        top = (img_height - new_height) / 2
        img = img.crop((0, top, img_width, top + new_height))

    # 4. Guardar la imagen recortada de nuevo en memoria para ReportLab
    img_io = BytesIO()
    img.save(img_io, format='PNG')
    img_io.seek(0)

    # 5. Retornar el objeto Platypus con el tamaño final exacto
    return Image(img_io, width=target_width, height=target_height)


# ------- Metodo que generara el PDF final de un reporte de gardclean -------
# 1. Definir el objeto "señal" legal para Platypus
class FooterTrigger(Flowable):
    def __init__(self):
        Flowable.__init__(self)
        self.width = 0
        self.height = 0

    def draw(self):
        # 'canv' es el acceso al canvas actual desde el Flowable
        self.canv.es_la_ultima = True

def later_pages_reporte_header(canvas, doc, logo_gardclean_ruta, logo_cliente_ruta, num_secciones_reporte_gardclean, signature_container_table):
    canvas.saveState()
    canvas.setFont("Arial_Bold",16)

    page_full_width = doc.pagesize[0]
    page_full_height = doc.pagesize[1]

    # Dibuja el logo de gardclean preserveAspectRatio=True
    canvas.drawImage(logo_gardclean_ruta, 15,page_full_height - 0.8 * inch,width=3.97 * cm,height=1.27 * cm,mask='auto')

    # Dibuja el logo del cliente
    canvas.drawImage(logo_cliente_ruta, (page_full_width - 1.6 * inch),page_full_height - 0.8 * inch,width=109.7,height=32.9,preserveAspectRatio=True,mask='auto')

    # Establece el footer de la ultima pagina
    canvas.setSignatureContainerTable(signature_container_table)

    canvas.restoreState()

def first_page_reporte_header(canvas,doc, logo_gardclean_ruta, logo_cliente_ruta, encabezado_paragraph):
    # Dimensiones de anchura y altura deseadas en las imagenes
    target_width = 100
    target_height = 50

    # Propiedades del documento
    page_full_width = doc.pagesize[0]
    page_full_height = doc.pagesize[1]

    # *** Logo de gardclean ***

    # -- Carga la imagen y lee sus dimensiones --
    logo_gardclean_img = ImageReader(logo_gardclean_ruta)
    logo_gardclean_img_width, logo_gardclean_img_height = logo_gardclean_img.getSize()

    # Calcula el factor escala para ajustar el tamaño de la imagen a el ancho y alto deseado manteniendo la relación de aspecto
    scale_factor_width = target_width / logo_gardclean_img_width
    scale_factor_height = target_height / logo_gardclean_img_height
    scale_factor = min(scale_factor_width, scale_factor_height)

    # Determina las nuevas dimensiones
    logo_gardclean_img_new_width = logo_gardclean_img_width * scale_factor
    logo_gardclean_img_new_height = logo_gardclean_img_height * scale_factor

    # *** Logo de cliente ***
    # -- Carga la imagen y lee sus dimensiones --
    logo_cliente_img = ImageReader(logo_cliente_ruta)
    logo_cliente_img_width, logo_cliente_img_height = logo_cliente_img.getSize()

    # Calcula el factor escala para ajustar el tamaño de la imagen a el ancho y alto deseado manteniendo la relación de aspecto
    scale_factor_width = target_width / logo_cliente_img_width
    scale_factor_height = target_height / logo_cliente_img_height
    scale_factor = min(scale_factor_width, scale_factor_height)

    # Determina las nuevas dimensiones
    logo_cliente_img_new_width = logo_cliente_img_width * scale_factor
    logo_cliente_img_new_height = logo_cliente_img_height * scale_factor

    # ** Dibuja logo gardclean **
    canvas.drawImage(logo_gardclean_ruta, 15,page_full_height - 0.7 * inch,width=logo_gardclean_img_new_width,height=logo_gardclean_img_new_height,mask='auto',anchor='c')

    # *** Dibuja el logo del cliente ***
    canvas.drawImage(logo_cliente_ruta, (page_full_width - 1.6 * inch),page_full_height - 0.8 * inch,width=logo_cliente_img_new_width,height=logo_cliente_img_new_height,preserveAspectRatio=True,mask='auto',anchor='c')

    # *** Dibuja encabezado ***
    width_wrap, height_wrap = encabezado_paragraph.wrap( (page_full_width - page_full_width / 2) + 40, page_full_height)
    # 2300
    # 2300 / 2 = 1150
    # 2300 - 1150 =
    encabezado_paragraph.drawOn(canvas, ( (page_full_width - page_full_width / 2) - (page_full_width / 4) ) - 20 , page_full_height - 1.2 * inch)

def last_page_footer(canvas, doc, signature_container_table):
    # Propiedades del documento
    page_full_width = doc.pagesize[0]
    page_full_height = doc.pagesize[1]

    signature_container_table.wrapOn(canvas, page_full_width, page_full_height)
    signature_container_table.drawOn(canvas, ( (page_full_width - page_full_width / 2) - (page_full_width / 4) ) - 20,page_full_height - 0.1 * inch)

# Define una clase de Spacer personalizada
class ConditionalSpacer(Spacer):
    def wrap(self, availWidth, availHeight):
        # Use the smaller of the specified height or the available height
        # Subtracting a small epsilon (1e-8) can prevent potential corner case errors
        height = min(self.height, availHeight - 1e-8)
        return (availWidth, height)

class MyFooterCanvas(canvas.Canvas):
    def __init__(self, *args, **kwargs):
        ## Subclass the ReportLab canvas class.
        #canvas.Canvas.__init__(self, *args, **kwargs)
        super().__init__(*args,**kwargs)
        ## Create an empty list to store the saved pages so you can keep track of them.
        self.signature_container_table = None
        self._saved_states = []

    # Inicializa el objeto Platypus de la tabla de las firmas como footer de la ultima hoja
    def setSignatureContainerTable(self, signature_container_table):
        self.signature_container_table = signature_container_table

    def showPage(self):
        """Override the showPage method"""

        # Mantenemos el rastro de las paginas que se han dibujado para saber posteriormente si llego a la ultima pagina
        self._saved_states.append(dict(self.__dict__))
        ## Start a new page.
        self._startPage()

    def drawFooter(self):
        self.saveState()

        # Propiedades del documento
        page_full_width = self._pagesize[0]
        page_full_height = self._pagesize[1]

        self.signature_container_table.wrapOn(self, page_full_width, page_full_height)
        #self.signature_container_table.drawOn(self, ( (page_full_width - page_full_width / 2) - (page_full_width / 4) ) - 20, page_full_height - 0.8 * inch)
        self.signature_container_table.drawOn(self, 0 , 0.5 * cm)


        self.restoreState()

    def save(self):
        for i, state in enumerate(self._saved_states):
            # Restauramos el estado de cada página
            self.__dict__.update(state)

            # Condición: Solo si es la última página
            if i == len(self._saved_states) - 1:
                self.drawFooter()

            super().showPage()

        super().save()

def generate_report_pdf(request):
    if request.method == "GET":
        # Obtiene id de reporte del cual se generara el PDF
        reporte_gardclean_id = request.GET.get("reporte_gardclean_id",None) or None

        reporte_gardclean = ReporteGardclean.objects.filter(reporte_gardclean_id=reporte_gardclean_id).first()

        if reporte_gardclean is None:
            return HttpResponse("No se encontro el reporte",status=HTTPStatus.INTERNAL_SERVER_ERROR)

        # -- Datos de PDF ---
        report_primary_color = "#284E13"
        sections_background_color = "#205E12"
        sections_outline_color = "#38751D"

        pdf_file_name = f"Reporte {reporte_gardclean.titulo}.pdf"
        document_title = "Reporte-gardclean"

        # ---- Datos del proyecto ---
        proyecto_gardclean = reporte_gardclean.proyecto_gardclean
        encabezado = proyecto_gardclean.encabezado
        oficina = proyecto_gardclean.oficina
        oficina_nombre = oficina.oficina_nombre

        cliente = proyecto_gardclean.cliente
        cliente_nombre = cliente.cli_nombre_cliente
        cliente_logo_path = f"{STATIC_FULL_URL_PATH}/{cliente.cli_logo_empresa}"

        # --- Ruta de logo de gardclean --
        gardclean_logo_path = f"{STATIC_PREFIX_PATH}/images/logo-gardclean.png"


        response = {
            "url_reporte": gardclean_logo_path,
            "nombre_reporte": "lol"
        }

        """"
        return JsonResponse(
            response
        )
        """


        # ---- Datos del reporte ---
        datos_generales = reporte_gardclean.datos_generales
        zona_reporte = datos_generales.zona
        tipo_zona_reporte = datos_generales.tipo_zona
        limites = datos_generales.limites
        desecho = datos_generales.desecho
        desecho_unidad = datos_generales.desecho_unidad
        superficie = datos_generales.superficie
        superficie_unidad = datos_generales.superficie_unidad

        fecha_intervencion = reporte_gardclean.fecha_intervencion
        num_vuelta = reporte_gardclean.num_vuelta

        mes_numerico_intervencion = fecha_intervencion.month
        # Formatea la fecha de intervencion
        fecha_intervencion = fecha_intervencion.strftime("%d DE 'B' DE %Y a las %I:%M:%S %p").upper()
        # Obtiene el mes en espanol
        meses = [
            "ENERO","FEBRERO","MARZO","ABRIL","MAYO","JUNIO","JULIO","AGOSTO","SEPTIEMBRE","OCTUBRE","NOVIEMBRE","DICIEMBRE"
        ]
        mes_espanol = meses[mes_numerico_intervencion - 1]

        fecha_intervencion = fecha_intervencion.replace("'B'",mes_espanol)

        # --- Secciones del reporte gardclean ---
        secciones_reporte_gardclean = SeccionReporteGardclean.objects.filter(reporte_gardclean=reporte_gardclean)


        # --- Empieza generacion de PDF ----

        # *** 1.- Crea objeto principal de PDF ***
        top_margin = 1.0 * inch
        bottom_margin = 0 * inch
        PAGE_HEIGHT = A4[1]
        PAGE_WIDTH = A4[0]
        spacer_full_height = PAGE_HEIGHT - top_margin - bottom_margin

        pdf = SimpleDocTemplate(
            pdf_file_name,
            pagesize=portrait(A4),
            #pagesize=letter,
            title = reporte_gardclean.titulo,
            author="Gardclean",
            subject = "Reporte generado por Gardclean.",

            leftMargin=0 * inch,
            rightMargin=0 * inch,
            #topMargin=0.50 * cm,
            topMargin = top_margin ,
            bottomMargin= bottom_margin
        )

        pdf.es_ultima_pagina = False # Inicialmente False

        # topMargin = 1.2 * inch ,

        page_width, page_height = A4

        """
        leftMargin=0.27 * cm,
        rightMargin=0.77 * cm,
        topMargin=0.91 * cm,
        bottomMargin=1.25 * cm
        """

        # 1.1.- Crea lista de elementos que se van a agregar al reporte PDF
        pdf_elements = []

        # 1.2.- Registra las fuentes que se usaran en el PDF
        pdfmetrics.registerFont(TTFont("Arial_Regular",f"{STATIC_PREFIX_PATH}/fonts/LiberationSans-Regular.ttf"))
        pdfmetrics.registerFont(TTFont("Arial_Bold",f"{STATIC_PREFIX_PATH}/fonts/LiberationSans-Bold.ttf"))

        # 1.3.- Obtenemos estilos predefinidos
        styles = getSampleStyleSheet()

        # Estilo para los textos con salto de linea
        preformatted_style = styles['Code']

        # *** 2.- Agrega cabecera ***

        # 2.1.- Tabla que representara los elementos de la cabecera
        header_table_data = [
            []
        ]

        # 2.1.1.- Logo de gardclean
        # kind='proportional' width=3.89 * cm, height=1.17 * cm
        gardclean_logo = Image(gardclean_logo_path)
        #gardclean_logo = Paragraph(f'<img src="{gardclean_logo_path}" width="50" height="50"/>')
        #gardclean_logo._restrictSize(3.89 * cm, 1.17 * cm)
        gardclean_logo._restrictSize(4.02 * cm, 1.24 * cm)

        # Agrega logo de gardclean a datos de tabla de cabecera
        header_table_data[0].append(gardclean_logo)

        # 2.1.2.- Encabezado principal de reporte
        encabezado_paragraph_style = ParagraphStyle(
            "centered",
            fontName="Arial_Regular",
            fontSize=8,
            parent=styles['Title'],
            alignment=1,
            textColor = "black",
            leading=11,
            #spaceAfter=60,
            #spaceBefore=10,


            #borderWidth=2,
            #borderColor=colors.black,
            #borderPadding=10,
            #borderRadius=5,

        )

        encabezado_formateado = encabezado.replace("\n","<br/>")

        #encabezado_paragraph = XPreformatted(encabezado, preformatted_style)

        encabezado_paragraph = Paragraph(
            encabezado_formateado,
            encabezado_paragraph_style
        )

        # Crea funcion partial y el enviara los datos necesarios para la cabecera de la primer pagina
        first_page_reporte_header_with_params = partial(first_page_reporte_header, logo_gardclean_ruta=gardclean_logo_path, logo_cliente_ruta=cliente_logo_path, encabezado_paragraph=encabezado_paragraph)

        """"
        # Agrega encabezado a datos de tabla de cabecera
        header_table_data[0].append(encabezado_paragraph)

        # 2.1.3.- Logo de cliente
        cliente_logo = Image(cliente_logo_path)
        cliente_logo._restrictSize(3.33 * cm, 1.65 * cm)

        # Agrega logo de cliente a datos de tabla de cabecera
        header_table_data[0].append(cliente_logo)


        # 2.2.- Agrega estilo a la tabla para acomodar los elementos
        header_table_style = TableStyle([
            # Fondo transparente
            #('BACKGROUND',(0,0),(-1,-1),colors.transparent),

            # Color de texto
            #('TEXTCOLOR',(0,0),(-1,-1),colors.transparent),

            # Alineacion horizontal
            ('ALIGN',(0,0),(-1,-1),'CENTER'),

            # Alineacion vertical
            ('VALIGN',(0,0),(-1,-1),'CENTER'),

            # Alineacion vertical logo cliente
            ('VALIGN',(2,0),(2,0),'TOP'),

            # Alineacion vertical logo gardclean
            ('VALIGN',(0,0),(0,0),'TOP'),

            # Alineacion horizontal logo gardclean
            ('ALIGN',(0,0),(0,0),'LEFT'),

            # Alineacion horizontal logo cliente
            ('ALIGN',(-1,-1),(-1,-1),'CENTER'),

            #('GRID',(0,0),(-1,-1),2,colors.green)
        ])
        """

        # header_table = Table(header_table_data,hAlign='CENTER',colWidths=[1.7*inch, None, 1.7*inch])
        #header_table = Table(header_table_data,hAlign='CENTER',colWidths=[1.7*inch, None, 1.7*inch])
        #header_table = Table(header_table_data,hAlign='CENTER',colWidths=[1.5*inch, None, 1.5*inch])

        #header_table.setStyle(header_table_style)



        # 2.3.- Agrega elemento de tabla de cabecera a la lista de elementos del PDF
        #pdf_elements.append(header_table)

        # *** --- ***

        # 2.4.- Agrega fecha y hora de intervencion y el num de vueltas
        fecha_intervencion_vueltas_paragraph_style = ParagraphStyle(
            "right",
            fontName="Arial_Bold",
            fontSize=9,
            parent=styles['Title'],
            alignment=2,
            textColor = "black",
            leading=5,
            rightIndent=0.4*inch,
            #spaceAfter=60,
            #spaceBefore=10
        )

        fecha_intervencion_paragraph = Paragraph(
            f"FECHA DE INTERVENCIÓN: {str(fecha_intervencion)}",
            fecha_intervencion_vueltas_paragraph_style
        )

        num_vueltas_paragraph = Paragraph(
            f"NO. VUELTA: {num_vuelta if num_vuelta is not None else ''}",
            fecha_intervencion_vueltas_paragraph_style
        )

        pdf_elements.append(
            Spacer(0.6 * cm, 0.6 * cm)
        )

        pdf_elements.append(
            fecha_intervencion_paragraph
        )

        if num_vuelta is not None:
            pdf_elements.append(num_vueltas_paragraph)


        # 2.5.- Agrega tabla de '1.-DATOS GENERALES'
        datos_generales_table_data = [
            []
        ]

        # 2.5.1.- Parrafo de cabecera de tabla
        datos_g_table_header_para_style = ParagraphStyle(
            "centered",
            fontName="Arial_Bold",
            fontSize=9,
            parent=styles['Normal'],
            alignment=1,
            textColor = colors.whitesmoke,
            #rightIndent=0.4*inch,
            #spaceAfter=60,
            #spaceBefore=10
        )
        datos_g_table_header_para = Paragraph(
            "1.- DATOS GENERALES",
            datos_g_table_header_para_style
        )

        # 2.5.2.- Agrega parrafo de cabecera
        datos_generales_table_data[0].append(datos_g_table_header_para)

        # 2.5.3.- Agrega parrafos de datos generales
        datos_generales_table_prefix_paragraph_style = ParagraphStyle(
            "left",
            fontName="Arial_Regular",
            fontSize=9,
            parent=styles['Normal'],
            alignment=0,
            textColor = colors.black,
            #rightIndent=0.4*inch,
            #spaceAfter=60,
            #spaceBefore=10
        )

        datos_generales_table_paragraph_style = ParagraphStyle(
            "left",
            fontName="Arial_Regular",
            fontSize=9,
            parent=styles['Normal'],
            alignment=0,
            textColor = colors.black,
            #leftIndent=0.1 * cm,
            #rightIndent=0.4*inch,
            #spaceAfter=60,
            #spaceBefore=10
        )


        zona_paragraph = Paragraph(
            f"{tipo_zona_reporte.nombre.upper()}: {zona_reporte.upper()}",
            datos_generales_table_paragraph_style
        )

        limites_paragraph = Paragraph(
            f"LIMITES: {limites.upper()}",
            datos_generales_table_paragraph_style
        )

        desecho_paragraph = Paragraph(
            f"DESECHO: {desecho} {desecho_unidad}",
            datos_generales_table_paragraph_style
        )

        superficie_paragraph = Paragraph(
            f"SUPERFICIE: {superficie} {superficie_unidad}",
            datos_generales_table_paragraph_style
        )

        datos_generales_table_data.append(
            [zona_paragraph]
        )
        datos_generales_table_data.append(
            [limites_paragraph]
        )
        datos_generales_table_data.append(
            [desecho_paragraph]
        )
        datos_generales_table_data.append(
            [superficie_paragraph]
        )


        """"
        zona_prefix_paragraph = Paragraph(
            f"{tipo_zona_reporte.nombre.upper()}: ",
            datos_generales_table_prefix_paragraph_style
        )

        limites_prefix_paragraph = Paragraph(
            f"LIMITES: ",
            datos_generales_table_prefix_paragraph_style
        )

        desecho_prefix_paragraph = Paragraph(
            f"DESECHO: ",
            datos_generales_table_prefix_paragraph_style
        )

        superficie_prefix_paragraph = Paragraph(
            f"SUPERFICIE: ",
            datos_generales_table_prefix_paragraph_style
        )

        zona_paragraph = Paragraph(
            f"{zona_reporte.upper()}",
            datos_generales_table_paragraph_style
        )

        limites_paragraph = Paragraph(
            f"{limites.upper()}",
            datos_generales_table_paragraph_style
        )

        desecho_paragraph = Paragraph(
            f"{desecho} {desecho_unidad}",
            datos_generales_table_paragraph_style
        )

        superficie_paragraph = Paragraph(
            f"{superficie} {superficie_unidad}",
            datos_generales_table_paragraph_style
        )

        zona_table_paragraph = Table([[zona_prefix_paragraph,zona_paragraph]], colWidths=[1.0 * inch])
        limites_table_paragraph = Table([[limites_prefix_paragraph,limites_paragraph]])
        desecho_table_paragraph = Table([[desecho_prefix_paragraph,desecho_paragraph]])
        superficie_table_paragraph = Table([[superficie_prefix_paragraph,superficie_paragraph]])

        datos_generales_table_data.append(
            [zona_table_paragraph]
        )
        datos_generales_table_data.append(
            [limites_table_paragraph]
        )
        datos_generales_table_data.append(
            [desecho_table_paragraph]
        )
        datos_generales_table_data.append(
            [superficie_table_paragraph]
        )
        """

        # 2.5.4.- Crea estilo de tabla
        datos_generales_table_style = TableStyle([
            # Fondo transparente
            #('BACKGROUND',(0,0),(-1,-1),colors.transparent),

            # Color de texto
            #('TEXTCOLOR',(0,0),(-1,-1),colors.transparent),

            # Alineacion horizontal
            ('ALIGN',(0,0),(-1,-1),'CENTER'),

            # Alineacion vertical
            #('VALIGN',(0,0),(-1,-1),'CENTER'),

            # Color de fondo en la cabecera
            ('BACKGROUND',(0,0),(-1,0),report_primary_color),

            # Color de texto en la cabecera
            ('TEXTCOLOR',(0,0),(-1,0),colors.whitesmoke),

            # Combinacion de celdas de cabecera
            ('SPAN',(0,0),(0,0)),

            # Combinacion de celdas de los datos generales
            ('SPAN',(0,1),(0,1)), # Prefijo Zona
            ('SPAN',(0,2),(0,2)), # Prefijo Limites
            ('SPAN',(0,3),(0,3)), # Prefijo Desecho
            ('SPAN',(0,4),(0,4)), # Prefijo Superficie

            # Borde de tabla
            ('BOX',(0,1),(-1,-1),0.5,colors.black),

            # Grilla interna
            ('INNERGRID', (0, 0), (-1, -1), 0.25, colors.transparent)

            #('GRID',(0,0),(-1,-1),2,colors.green)
        ])

        # 2.5.5.- Crea tabla
        datos_generales_table = Table(datos_generales_table_data,colWidths=None)
        datos_generales_table.setStyle(datos_generales_table_style)

        # 2.5.6.- Agrega tabla a PDF
        pdf_elements.append(
            Spacer(
                0.4*inch,0.4*inch
            )
        )
        pdf_elements.append(
            datos_generales_table
        )

        # *** 2.6.- Agrega mapa de google de los limites ***

        google_static_map_url = get_static_map_by_address_local(
           address=limites,
           size="640x640",
           zoom="14",
           scale="2",
           styles=[
               "style=feature:poi|visibility:off",
               #"style=feature:administrative|element:labels|visibility:off"
               #"style=feature:landscape|visibility:off"
           ]
        )

        #google_static_map_url = "https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcSnYtM5__fKPJgkLLdc1LVgxq0Ycm_JzIiygg&s"
        google_static_map_url = urllib.parse.quote(google_static_map_url,safe=":/?=&")
        google_static_map_url = google_static_map_url.replace("&amp;","&")
        # width=19.84 * cm,height=10.67 * cm
        #google_static_map_image = Image(google_static_map_url,width=19.84 * cm,height=7.67 * cm)
        google_static_map_image = get_static_map_img_cropped(
            google_static_map_url,
            PAGE_WIDTH * 0.95,
            10*cm
        )
        """
        google_static_map_image = get_reportlab_image(
            img_path=google_static_map_url,
            width_percent=95,
            height=10*cm
        )
        """

        #google_static_map_image._restrictSize(19.84 * cm, 10.67 * cm)
        pdf_elements.append(
            KeepTogether(
                Spacer(
                    0.2*inch,0.2*inch
                )
            )
        )
        pdf_elements.append(
            google_static_map_image
        )
        pdf_elements.append(
            KeepTogether(
                Spacer(
                    0.1*inch,0.1*inch
                )
            )
        )

        # *** 2.7.-Agrega las secciones de imagenes de evidencia ***

        # 2.7.1.- Crea el estilo de las cabeceras de las secciones como una tabla
        section_header_paragraph_style = ParagraphStyle(
            "centered",
            fontName="Arial_Bold",
            fontSize=9,
            parent=styles['Normal'],
            alignment=1,
            textColor = colors.whitesmoke,
            backColor = sections_background_color
            #leading=4,
            #spaceAfter=60,
            #spaceBefore=10
        )

        section_header_table_data = [
            [
                Paragraph(
                    "",
                    section_header_paragraph_style
                )
            ]
        ]

        section_header_table_style = TableStyle([
            # Borde de tabla
            ('LINEABOVE',(0,0),(-1,0),1.5,sections_outline_color),
            ('LINEBEFORE',(0,0),(-1,0),1.5,sections_outline_color),
            ('LINEAFTER',(0,0),(-1,0),1.5,sections_outline_color),


            # Color de fondo en la cabecera
            #('BACKGROUND',(0,0),(-1,0),report_primary_color),

            # Padding top 2
            ('TOPPADDING',(0,0),(-1,0),2),
            ('LEFTPADDING',(0,0),(-1,0),0),
            ('RIGHTPADDING',(0,0),(-1,0),0),
            ('BOTTOMPADDING',(0,0),(-1,0),1.5),
        ])

        #section_header_table = Table()

        # 2.7.2.- Itera sobre todas las secciones del reporte e muestra las imagenes de evidencia de cada sección
        seccion_reporte_gc_contador = 0
        for seccion_reporte_gc in secciones_reporte_gardclean:
            # Obtiene datos de la seccion
            seccion_proyecto_gc = seccion_reporte_gc.seccion_proyecto_gc
            titulo_seccion = seccion_proyecto_gc.titulo.replace("[]",tipo_zona_reporte.nombre).upper()
            num_img_evidencia = seccion_proyecto_gc.num_img_evidencia

            # Crea parrafo de titulo de la seccion
            section_header_table_data[0][0] = Paragraph(
                titulo_seccion,
                section_header_paragraph_style
            )

            # Tabla del titulo de la sección
            section_header_table = Table(section_header_table_data)
            section_header_table.setStyle(section_header_table_style)

            # *** CADA VEZ QUE VAYA A INSERTAR UNA SECCIÓN (SIN CONTAR LA PRIMERA), INSERTARA UN SALTO DE PAGINA MIENTRAS
            # LA SECCION NO SEA UNA QUE NO TENGA IMAGENES DE EVIDENCIA ***
            seccion_anterior_proyecto_gc = secciones_reporte_gardclean[seccion_reporte_gc_contador - 1].seccion_proyecto_gc if seccion_reporte_gc_contador > 0 else seccion_proyecto_gc
            if (seccion_reporte_gc_contador != 0 and seccion_anterior_proyecto_gc.num_img_evidencia is not None):
                pdf_elements.append(
                    PageBreak()
                )

            # Spacer (SOLO APLICA EN LA PRIMER SECCION)
            if seccion_reporte_gc_contador == 0:
                pdf_elements.append(
                    KeepTogether(
                        Spacer(0.2 * inch, 0.2 * inch)
                    )
                )

            pdf_elements.append(section_header_table)

            # Spacer
            pdf_elements.append(
                KeepTogether(
                    Spacer(0.2*inch,0.2*inch)
                )
            )


            # Obtiene imagenes de evidencia de la sección
            imagenes_evidencia_seccion_rgc = ImagenEvidenciaRg.objects.filter(seccion_reporte_gardclean=seccion_reporte_gc)

            # Crea tabla de imagenes para la seccion
            img_evidencia_table_data = []

            img_evidencia_table_style = TableStyle([
                # Grilla interna
                #('INNERGRID', (0, 0), (-1, -1), 0.25, colors.transparent),

                # Borde de la tabla
                #('BOX',(0,0),(-1,-1),1,colors.transparent),

                # Padding
                #('RIGHTPADDING',(0,0),(0,-1),8),
                #('LEFTPADDING',(1,0),(-1,-1),28)
                ('RIGHTPADDING',(1,0),(1,-1), 50),
                ('LEFTPADDING',(1,0),(1,-1), 55),

                ('TOPPADDING',(0,1),(-1,-1),0) # 50

            ])

            row_count = 0 # Contador que indicara si ya se han completado las 3 imagenes por fila
            table_data_row = 0 # Indice que indica el numero de fila en los datos de la tabla
            row_inserted = False # Bandera boolean que indica si ya se inserto la nueva fila en los datos de la tabla
            for evidence_image_record in imagenes_evidencia_seccion_rgc:
                #return HttpResponse(len(imagenes_evidencia_seccion_rgc))
                evidence_image = Image(f"{STATIC_FULL_URL_PATH}/{evidence_image_record.ruta}")
                evidence_image._restrictSize(4.75 * cm, 6.35 * cm)

                # Si ya mostro 3 imagenes en una fila, las otras 3 siguientes las pondra en la siguiente fila
                if row_count % 3 == 0 and row_count != 0:
                    table_data_row += 1 # Aumenta en uno el indice de la fila actual en los datos de la tabla
                    row_count = 0 # Reinicia el contador de las 3 imagenes por fila
                    row_inserted = False # Reinicia la bandera para que pueda insertar otra fila

                # A los datos de la tabla le agrega otra fila(solo si no la ha insertado antes de saltar a la otra fila)
                if not row_inserted:
                    img_evidencia_table_data.append([])
                    row_inserted = True


                # Crea una tabla que contendra la imagen de evidencia y su posible descripcion
                img_evidencia_description_paragraph_style = ParagraphStyle(
                    "centered",
                    fontName="Arial_Bold",
                    fontSize=7,
                    parent=styles['Normal'],
                    alignment=0,
                    textColor = colors.black,
                    spaceBefore=25
                )

                img_evidencia_description_table = Table(
                    [
                        [evidence_image],
                        [
                            Paragraph(
                                evidence_image_record.descripcion,
                                img_evidencia_description_paragraph_style
                            )
                        ]
                    ],
                    colWidths=[4.75 * cm],
                    #rowHeights=[6.35 * cm,2*cm]
                    rowHeights=[6.35*cm,1.2*cm]
                )

                if evidence_image_record.descripcion == "" or evidence_image_record.descripcion is None:
                    pass
                    img_evidencia_description_table.rowHeights = [6*35*cm,None]

                img_evidencia_description_table.setStyle(
                    TableStyle([
                        #('BOX',(0,0),(-1,-1),0.75,colors.black),
                        #('INNERGRID',(0,0),(-1,-1),0.75,colors.black),
                        ('VALIGN',(0,0),(-1,-1),'TOP'),
                        ('VALIGN',(0,1),(0,1),'MIDDLE')
                    ])
                )

                """"
                img_evidencia_table_data[table_data_row].append(
                    #evidence_image
                    img_evidencia_description_table
                )
                """
                img_evidencia_table_data[table_data_row].append(
                    #evidence_image
                    img_evidencia_description_table
                )

                row_count += 1


            if (len(img_evidencia_table_data) > 0):
                img_evidencia_table = Table(img_evidencia_table_data)
                img_evidencia_table.hAlign = 'CENTER'
                img_evidencia_table.setStyle(
                    img_evidencia_table_style
                )

                pdf_elements.append(img_evidencia_table)

            seccion_reporte_gc_contador += 1

        #pdf_elements.append(gardclean_logo)

        # *** 2.8.- Agrega elementos de firmas ***
        signature_paragraph_style = ParagraphStyle(
            "centered",
            fontName="Arial_Regular",
            fontSize=8,
            parent=styles['Normal'],
            alignment=1,
            textColor = colors.black,
            spaceBefore=25
        )

        signature_container_table_data = [
            []
        ]

        signature_container_table_style = TableStyle([
            ('INNERGRID',(0,0),(-1,-1),0.25,colors.transparent),
            ('BOX',(0,0),(-1,-1),0.25,colors.transparent)
        ])

        signature_table_style = TableStyle([
            ('INNERGRID',(0,0),(-1,-1),0.25,colors.transparent),
            ('BOX',(0,0),(-1,-1),0.25,colors.transparent),

            ('TOPPADDING',(0,1),(0,1),10)
        ])

        # 2.8.1.- Crea primer elemento de firma
        first_signature_drawing = Drawing(150, 1)
        first_signature_drawing.add(Line(0, 0, 150, 0))

        first_signature_text = proyecto_gardclean.primer_firma
        first_signature_text = first_signature_text.replace("\n","<br/>")

        first_signature_paragraph = Paragraph(
            #"C. José Roberto Valadez Ramírez <br/>Representante Legal<br/>Gard Clean",
            first_signature_text,
            signature_paragraph_style
        )

        first_signature_table_data = [
            [first_signature_drawing],
            [first_signature_paragraph]
        ]

        first_signature_table = Table(
            first_signature_table_data
        )

        first_signature_table.setStyle(
            signature_table_style
        )

        # 2.8.2.- Crea segundo elemento de firma
        second_signature_drawing = Drawing(150, 1)
        second_signature_drawing.add(Line(0, 0, 150, 0))

        second_signature_text = proyecto_gardclean.segunda_firma
        second_signature_text = second_signature_text.replace("\n","<br/>")

        second_signature_paragraph = Paragraph(
            #"Lic. Miriam Lucia Cerna Fernández <br/>Dirección<br/>Parques y Jardines",
            second_signature_text,
            signature_paragraph_style
        )

        second_signature_table_data = [
            [second_signature_drawing],
            [second_signature_paragraph]
        ]

        second_signature_table = Table(
            second_signature_table_data
        )

        second_signature_table.setStyle(
            signature_table_style
        )

        # 2.8.3.- Agrega los elementos de firma al contenedor principal
        signature_container_table_data[0].append(
            Spacer(0.1*inch,0.1*inch)
        )

        signature_container_table_data[0].append(
            first_signature_table
        )

        signature_container_table_data[0].append(
            Spacer(0.1*inch,0.1*inch)
        )

        signature_container_table_data[0].append(
            second_signature_table
        )

        signature_container_table_data[0].append(
            Spacer(0.1*inch,0.1*inch)
        )

        signature_container_table = Table(
            signature_container_table_data
            #signature_container_table_style
        )

        signature_container_table.setStyle(
            signature_container_table_style
        )

        pdf.signature_container_table = signature_container_table

        # 2.8.4.- Agrega el elemento principal de firmas al PDF
        """
        pdf_elements.append(
            KeepTogether(
                [
                    ConditionalSpacer(
                        width=1*cm, height=1*cm
                    ),
                    signature_container_table
                ]
            )
        )
        """


        """
        pdf_elements.append(
            PageBreak()
        )

        pdf_elements.append(
            KeepTogether(
                Spacer(
                    0.1 * inch, 0.1 * inch
                )
            )
        )
        pdf_elements.append(
            signature_container_table
        )
        """

        later_pages_reporte_header_with_params = partial(later_pages_reporte_header, logo_gardclean_ruta=gardclean_logo_path, logo_cliente_ruta=cliente_logo_path, num_secciones_reporte_gardclean=len(secciones_reporte_gardclean), signature_container_table=signature_container_table)

        #pdf_elements.append(FooterTrigger())

        #last_page_footer_with_params = partial(last_page_footer,signature_container_table=signature_container_table)
        # onLastPage=last_page_footer_with_params
        pdf.build(pdf_elements,onFirstPage=first_page_reporte_header_with_params,onLaterPages=later_pages_reporte_header_with_params,canvasmaker=MyFooterCanvas)

        # Retornara una respuesta que sera el archivo generado para que el usuario
        # pueda descargarlo
        with open(pdf_file_name, "rb") as file:

            # Verifica si existe un archivo existente del reporte, de ser asi, lo eliminara
            ruta_reporte_actual = f"{STATIC_PREFIX_PATH}/{reporte_gardclean.ruta_reporte_generado}"

            if os.path.isfile(ruta_reporte_actual):
                os.remove(ruta_reporte_actual)

            # --- Guarda el PDF en la ruta donde se guardara el reporte PDF ---
            reporte_generado_ruta = upload_file_to_server(
                folder_name="reportes-gardclean-generados",
                file_name=document_title,
                file=file,
            )

            # Actualiza la ruta donde esta guardado el PDF final generado
            reporte_gardclean.ruta_reporte_generado = reporte_generado_ruta
            reporte_gardclean.save()

            response = {
                "url_reporte": f"{STATIC_FULL_URL_PATH}/{reporte_generado_ruta}",
                "nombre_reporte": f"{pdf_file_name}"
            }

            return JsonResponse(response)

            #response = HttpResponse(file.read(),content_type='application/pdf')

            # Agrega el nombre del archivo
            #response['Content-Disposition'] = f'attachment; filename={pdf_file_name}'

            #os.remove(file_name)

            return response



    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def delete_gardclean_report(request):
    if request.method == "POST":
        response = {
            "success": False,
            "error": False,
            "error_message": ""
        }
        # Obtiene cuerpo de la peticion
        body = json.loads(request.body)
        # Obtiene los datos individuales de la peticion
        reporte_gardclean_id = body["reporteGardcleanId"]

        # Obtiene registro principal de reporte
        reporte_gardclean = ReporteGardclean.objects.filter(reporte_gardclean_id=reporte_gardclean_id).first()

        if reporte_gardclean is None:
            return HttpResponse("No se encontro el reporte",status=HTTPStatus.METHOD_NOT_ALLOWED)

        # Obtiene las secciones propias del reporte de gardclean
        secciones_reporte_gardclean = SeccionReporteGardclean.objects.filter(reporte_gardclean=reporte_gardclean)
        for seccion in secciones_reporte_gardclean:
            # Obtiene las iamgenes de evidencia de la seccion
            imagenes_evidencia_seccion_r_gc = ImagenEvidenciaRg.objects.filter(seccion_reporte_gardclean=seccion)
            for img_evidencia in imagenes_evidencia_seccion_r_gc:
                ruta = f"{STATIC_PREFIX_PATH}/{img_evidencia.ruta}"
                if os.path.isfile(ruta):
                    os.remove(ruta)
                img_evidencia.delete()
            seccion.delete()

        reporte_gardclean.delete()

        response["success"] = True

        return JsonResponse(
            response
        )

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

def get_gardclean_report_by_id(request):
    if request.method == "GET":
        response = {
            "reporte_gardclean": {}
        }

        reporte_gardclean_id = request.GET.get("reporte_gardclean_id",None) or None

        reporte_gardclean = ReporteGardclean.objects.filter(reporte_gardclean_id=reporte_gardclean_id).first()
        if reporte_gardclean is None:
            return HttpResponse("No se encontro el reporte de gardclean",status=HTTPStatus.INTERNAL_SERVER_ERROR)

        reporte_datos_generales = reporte_gardclean.datos_generales
        proyecto_gardclean = reporte_gardclean.proyecto_gardclean

        response["reporte_gardclean"] = {
            "reporte_id": reporte_gardclean.reporte_gardclean_id,
            "titulo": reporte_gardclean.titulo,
            #"num_adquisicion": reporte_gardclean.num_adquisicion,
            #"requirente": reporte_gardclean.requirente,
            "datos_generales": {
                "tipo_zona": reporte_datos_generales.tipo_zona.nombre,
                "zona": reporte_datos_generales.zona,
                "limites": reporte_datos_generales.limites,
                "desecho": reporte_datos_generales.desecho,
                "desecho_unidad": reporte_datos_generales.desecho_unidad,
                "superficie": reporte_datos_generales.superficie,
                "superficie_unidad": reporte_datos_generales.superficie_unidad
            },
            "proyecto_gardclean": {
                "proyecto_gardclean_id": proyecto_gardclean.proyecto_gardclean_id,
                "titulo": proyecto_gardclean.titulo,
                "encabezado": proyecto_gardclean.encabezado,
                "nombre_oficina": proyecto_gardclean.oficina.oficina_nombre,
                "nombre_cliente": proyecto_gardclean.cliente.cli_nombre_cliente,
                "id_cliente": proyecto_gardclean.cliente.cli_id
            },
            "fecha_intervencion": str(reporte_gardclean.fecha_intervencion),
            "numero_vuelta": str(reporte_gardclean.num_vuelta)
        }

        return JsonResponse(
            response
        )

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

def get_evidence_images_by_section(request):
    if request.method == "GET":
        response = {
            "imagenes_evidencia": []
        }

        seccion_reporte_gc_id = request.GET.get("seccion_reporte_gc_id")
        seccion_reporte_gc_obj = SeccionReporteGardclean.objects.filter(seccion_reporte_gardclean_id=seccion_reporte_gc_id).first()

        if seccion_reporte_gc_obj is None:
            return HttpResponse("No se encontro la seccion",status=HTTPStatus.INTERNAL_SERVER_ERROR)

        imagenes_evidencia = ImagenEvidenciaRg.objects.filter(seccion_reporte_gardclean=seccion_reporte_gc_obj)

        response["imagenes_evidencia"] = [
            {
                "archivo_url": f"{STATIC_FULL_URL_PATH}/{img_e.ruta}",
                "descripcion": img_e.descripcion
            }
            for img_e in imagenes_evidencia
        ]

        return JsonResponse(
            response
        )

    return HttpResponse(status=HTTPStatus.IM_A_TEAPOT)

@csrf_exempt
def get_sections_by_gardclean_report(request):
    if request.method == "GET":
        response = {
            "secciones": []
        }

        # Obtiene parametros
        reporte_gardclean_id = request.GET.get("reporte_gardclean_id",None) or None

        # Obtiene objeto de reporte de gardclean
        reporte_gardclean = ReporteGardclean.objects.filter(reporte_gardclean_id=reporte_gardclean_id).first()

        if reporte_gardclean is None:
            return HttpResponse("No se encontro el reporte",status=HTTPStatus.INTERNAL_SERVER_ERROR)

        secciones = SeccionReporteGardclean.objects.filter(reporte_gardclean=reporte_gardclean).order_by("seccion_reporte_gardclean_id")

        response["secciones"] = [
            {
                "seccion_reporte_gardclean_id": seccion.seccion_reporte_gardclean_id,
                "nombre": seccion.seccion_proyecto_gc.titulo,
                "num_img_evidencia": seccion.seccion_proyecto_gc.num_img_evidencia
            }
            for seccion in secciones
        ]

        return JsonResponse(response)

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def get_all_gardclean_reports_zones(request):
    if request.method == "GET":
        response = {
            "zonas": []
        }
        zonas_reporte_gc = ZonaReporteGc.objects.order_by("nombre")

        response["zonas"] = [
            zona.nombre
            for zona in zonas_reporte_gc
        ]

        return JsonResponse(response)

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)


def get_gardclean_reports_zones_by_client(request):
    if request.method == "GET":
        response = {
            "zonas": []
        }

        # Obtiene el id del cliente
        cliente_id = request.GET.get("cliente_id",None) or None
        cliente = Clientes.objects.filter(cli_id=cliente_id).first()

        clientes_zonas_reportes_gc = ClienteZonaReporteGc.objects.filter(cliente=cliente).order_by("zona_reporte_gc__nombre")

        response["zonas"] = [
            zona.zona_reporte_gc.nombre
            for zona in clientes_zonas_reportes_gc
        ]

        return JsonResponse(response)
    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def get_all_gardclean_projects(request):
    if request.method == "GET":
        response = {
            "proyectos_gardclean": []
        }

        query = Q(
            estatus_proyecto_gardclean__estatus_proyecto_gardclean_id=2
        )

        proyectos_gardclean = ProyectoGardclean.objects.filter(~query).order_by("titulo")

        response["proyectos_gardclean"] = [
            {
                "proyecto_gardclean_id": proyecto.proyecto_gardclean_id,
                "titulo": proyecto.titulo,
                "encabezado": proyecto.encabezado,
                "nombre_oficina": proyecto.oficina.oficina_nombre,
                "nombre_cliente": proyecto.cliente.cli_nombre_cliente,
                "id_cliente": proyecto.cliente.cli_id
            }
            for proyecto in proyectos_gardclean
        ]

        return JsonResponse(
            response
        )

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

def get_gardclean_projects_by_offices(request):
    if request.method == "GET":
        response = {
            "proyectos_gardclean": []
        }

        oficinas_id = request.GET.getlist("oficinas_id")

        query = Q()

        for oficina_id in oficinas_id:
            query.add(Q(oficina__oficina_id=oficina_id),Q.OR)

        # Filtro de estatus de proyecto finalizado
        estatus_query = Q(
            estatus_proyecto_gardclean__estatus_proyecto_gardclean_id=2
        )

        proyectos_gardclean = ProyectoGardclean.objects.filter(query & ~estatus_query).order_by("titulo")

        response["proyectos_gardclean"] = [
            {
                "proyecto_gardclean_id": proyecto.proyecto_gardclean_id,
                "titulo": proyecto.titulo,
                "encabezado": proyecto.encabezado,
                "nombre_oficina": proyecto.oficina.oficina_nombre,
                "nombre_cliente": proyecto.cliente.cli_nombre_cliente,
                "id_cliente": proyecto.cliente.cli_id
            }
            for proyecto in proyectos_gardclean
        ]

        return JsonResponse(
            response
        )


    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def get_gardclean_reports(request):
    if request.method == 'GET':
        response = {
            "reportes": []
        }

        reportes_gardclean = ReporteGardclean.objects.order_by("-created_at")

        response["reportes"] = [
            {
                "reporte_id": reporte.reporte_gardclean_id,
                "titulo": reporte.titulo,
                #"num_adquisicion": reporte.num_adquisicion,
                #"requirente": reporte.requirente,
                "datos_generales": {
                    "tipo_zona": reporte.datos_generales.tipo_zona.nombre,
                    "zona": reporte.datos_generales.zona,
                    "limites": reporte.datos_generales.limites,
                    "desecho": reporte.datos_generales.desecho,
                    "desecho_unidad": reporte.datos_generales.desecho_unidades,
                    "superficie": reporte.datos_generales.superficie,
                    "superficie_unidad": reporte.datos_generales.superficie_unidad
                },
                "proyecto_gardclean": None,
                "fecha_intervencion": str(reporte.fecha_intervencion)

            }
            for reporte in reportes_gardclean
        ]

        return JsonResponse(response)

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)


def get_gardclean_reports_by_project(request):
    if request.method == "GET":
        response = {
            "reportes_gardclean": [],
            "num_registros": 0
        }

        proyecto_gardclean_id = request.GET.get("proyecto_gardclean_id",None) or None

        proyecto_gardclean_obj = ProyectoGardclean.objects.filter(proyecto_gardclean_id=proyecto_gardclean_id).first()

        if proyecto_gardclean_obj is None:
            return HttpResponse("No se encontro el proyecto", status = HTTPStatus.INTERNAL_SERVER_ERROR)

        reportes_gardclean = ReporteGardclean.objects.filter(proyecto_gardclean=proyecto_gardclean_obj).order_by("-created_at")

        response["reportes_gardclean"] = [
            {
                "reporte_id": reporte.reporte_gardclean_id,
                "titulo": reporte.titulo,
                #"num_adquisicion": reporte.num_adquisicion,
                #"requirente": reporte.requirente,
                "datos_generales": {
                    "tipo_zona": reporte.datos_generales.tipo_zona.nombre,
                    "zona": reporte.datos_generales.zona,
                    "limites": reporte.datos_generales.limites,
                    "desecho": reporte.datos_generales.desecho,
                    "desecho_unidad": reporte.datos_generales.desecho_unidad,
                    "superficie": reporte.datos_generales.superficie,
                    "superficie_unidad": reporte.datos_generales.superficie_unidad
                },
                "proyecto_gardclean": None,
                "fecha_intervencion": str(reporte.fecha_intervencion)

            }
            for reporte in reportes_gardclean
        ]
        response["num_registros"] = reportes_gardclean.count()

        return JsonResponse(response)

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def create_report(request):
    if request.method == 'POST':
        response = {
            "success": False,
            "error": False,
            "error_message": "",
            "report_id": None
        }
        # Obtiene cuerpo de la peticion
        body = json.loads(request.body)
        # Obtiene los datos individuales de la peticion
        proyecto_id = body["proyectoGardcleanId"]
        titulo = body["titulo"]
        numero_vuelta = body["numeroVuelta"]
        num_adquisicion = body["numAdquisicion"]
        requirente = body["requirente"]
        fecha_intervencion = body["fechaIntervencion"]

        #camellon = body["camellon"]

        zona = body["zona"]
        tipoZona = body["tipoZona"]
        limites = body["limites"]
        desecho = body["desecho"]
        desecho_unidad = body["desechoUnidad"]
        superficie = body["superficie"]
        superficie_unidad = body["superficieUnidad"]

        # Obtiene los objetos de los datos de la petición
        tipo_zona_obj = ZonaReporteGc.objects.get(nombre=tipoZona)
        proyecto_gardclean_obj = ProyectoGardclean.objects.get(proyecto_gardclean_id=proyecto_id)

        # Crea el registro pricipal del reporte junto con su registro de datos generales
        datos_generales = DatosGeneralesReporteG()
        #datos_generales.camellon = camellon
        datos_generales.tipo_zona = tipo_zona_obj
        datos_generales.zona = zona
        datos_generales.limites = limites
        datos_generales.desecho = desecho
        datos_generales.desecho_unidad = desecho_unidad
        datos_generales.superficie = superficie
        datos_generales.superficie_unidad = superficie_unidad
        datos_generales.save()

        reporte_gardclean = ReporteGardclean()
        #reporte_gardclean.tipo_zona_reporte_gc = tipo_zona_obj
        reporte_gardclean.proyecto_gardclean = proyecto_gardclean_obj
        reporte_gardclean.titulo = titulo
        reporte_gardclean.num_adquisicion = num_adquisicion
        reporte_gardclean.requirente = requirente
        reporte_gardclean.fecha_intervencion = datetime.strptime(fecha_intervencion, "%Y-%m-%d %H:%M:%S")
        reporte_gardclean.num_vuelta = numero_vuelta if numero_vuelta != -1 and numero_vuelta != "-1" else None

        reporte_gardclean.datos_generales = datos_generales
        reporte_gardclean.save()

        # Crea las secciones propias para la seccion de acuerdo al proyecto al que pertenece
        secciones_proyecto_gc = SeccionProyectoGc.objects.filter(proyecto_gardclean=proyecto_gardclean_obj)
        for seccion_proyecto_gc in secciones_proyecto_gc:
            seccion_reporte_gardclean = SeccionReporteGardclean()
            seccion_reporte_gardclean.reporte_gardclean = reporte_gardclean
            seccion_reporte_gardclean.seccion_proyecto_gc = seccion_proyecto_gc
            seccion_reporte_gardclean.save()

        response["success"] = True
        response["report_id"] = reporte_gardclean.reporte_gardclean_id

        return JsonResponse(response)

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def modify_report(request):
    if request.method == 'POST':
        response = {
            "success": False,
            "error": False,
            "error_message": "",
        }
        # Obtiene cuerpo de la peticion
        body = json.loads(request.body)
        # Obtiene los datos individuales de la peticion
        reporte_gardclean_id = body["reporteGardcleanId"]
        proyecto_id = body["proyectoGardcleanId"]
        titulo = body["titulo"]
        num_adquisicion = body["numAdquisicion"]
        requirente = body["requirente"]
        fecha_intervencion = body["fechaIntervencion"]



        #camellon = body["camellon"]

        zona = body["zona"]
        tipoZona = body["tipoZona"]
        limites = body["limites"]
        desecho = body["desecho"]
        desecho_unidad = body["desechoUnidad"]
        superficie = body["superficie"]
        superficie_unidad = body["superficieUnidad"]

        # Obtiene los objetos de los datos de la petición
        tipo_zona_obj = ZonaReporteGc.objects.get(nombre=tipoZona)
        proyecto_gardclean_obj = ProyectoGardclean.objects.get(proyecto_gardclean_id=proyecto_id)


        # ** Obtiene registro de reporte existente **
        reporte_gardclean = ReporteGardclean.objects.get(reporte_gardclean_id=reporte_gardclean_id)
        datos_generales = reporte_gardclean.datos_generales

        # Crea el registro pricipal del reporte junto con su registro de datos generales
        #datos_generales.camellon = camellon
        datos_generales.tipo_zona = tipo_zona_obj
        datos_generales.zona = zona
        datos_generales.limites = limites
        datos_generales.desecho = desecho
        datos_generales.desecho_unidad = desecho_unidad
        datos_generales.superficie = superficie
        datos_generales.superficie_unidad = superficie_unidad
        datos_generales.save()

        #reporte_gardclean.tipo_zona_reporte_gc = tipo_zona_obj
        reporte_gardclean.proyecto_gardclean = proyecto_gardclean_obj
        reporte_gardclean.titulo = titulo
        reporte_gardclean.num_adquisicion = num_adquisicion
        reporte_gardclean.requirente = requirente
        reporte_gardclean.fecha_intervencion = datetime.strptime(fecha_intervencion, "%Y-%m-%d %H:%M:%S")

        #response["error_message"] = str(reporte_gardclean.fecha_intervencion)
        #return JsonResponse(
        #    response
        #)

        #reporte_gardclean.datos_generales = datos_generales
        reporte_gardclean.save()

        # Crea las secciones propias para la seccion de acuerdo al proyecto al que pertenece
        """"
        secciones_proyecto_gc = SeccionProyectoGc.objects.filter(proyecto_gardclean=proyecto_gardclean_obj)
        for seccion_proyecto_gc in secciones_proyecto_gc:
            seccion_reporte_gardclean = SeccionReporteGardclean()
            seccion_reporte_gardclean.reporte_gardclean = reporte_gardclean
            seccion_reporte_gardclean.seccion_proyecto_gc = seccion_proyecto_gc
            seccion_reporte_gardclean.save()
        """

        response["success"] = True

        return JsonResponse(response)

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)


@csrf_exempt
def save_report_section(request):
    if request.method == 'POST':
        response = {
            "success": False,
            "error": False,
            "error_message": ""
        }
        # Obtiene cuerpo de la peticion
        body = json.loads(request.POST.get("body"))
        # Obtiene los datos individuales de la peticion
        reporte_id = body["reporteId"]
        titulo = body["titulo"]
        num_imagenes_evidencia = body["numImagenesEvidencia"]
        descripciones_imagenes_evidencia = body["descripcionesImagenesEvidencia"]

        # Obtiene las imagenes de evidencia de la seccion que se va a guardar
        imagenes_evidencias = request.FILES.getlist("imagenes-evidencia[]")

        #return HttpResponse(f"NUM: {num_imagenes_evidencia}")


        #return HttpResponse(len(imagenes_evidencias))

        # Obtiene el registro del reporte principal
        reporte_gardclean = ReporteGardclean.objects.filter(reporte_gardclean_id=reporte_id).first()

        if reporte_gardclean is None:
            return HttpResponse("None", status=HTTPStatus.INTERNAL_SERVER_ERROR)


        # Crea el registro principal de la seccion del reporte
        seccion_reporte_gc = SeccionReporteGardclean()
        seccion_reporte_gc.titulo = titulo
        seccion_reporte_gc.reporte_gardclean = reporte_gardclean
        seccion_reporte_gc.save()

        # Sube las imagenes al servidor y crea los registros de cada imagen asociados a la seccion con su descripcion correspondiente
        for i in range(0, num_imagenes_evidencia):
            #return HttpResponse(f"NUM: {num_imagenes_evidencia}")
            descripcion = descripciones_imagenes_evidencia[i]
            file = imagenes_evidencias[i]

            # Sube imagen al servidor
            ruta = upload_image_to_server(
                folder_name="img-evidencias-reportes-gc",
                image_name="img-evidencia-rg",
                image_file=file
            )

            # Crea registro en base de datos
            imagen_evidencia_rg = ImagenEvidenciaRg()
            imagen_evidencia_rg.ruta = ruta
            imagen_evidencia_rg.descripcion = descripcion
            imagen_evidencia_rg.seccion_reporte_gardclean = seccion_reporte_gc
            imagen_evidencia_rg.save()

        response["success"] = True

        return JsonResponse(response)


    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def save_report_section_changes(request):
    if request.method == "POST":
        response = {
            "success": False,
            "error": False,
            "error_message": ""
        }
        # Obtiene cuerpo de la peticion
        body = json.loads(request.POST.get("body"))
        # Obtiene los datos individuales de la peticion
        seccion_reporte_gc_id = body["seccionReporteGcId"]
        num_imagenes_evidencia = body["numImagenesEvidencia"]
        descripciones_imagenes_evidencia = body["descripcionesImagenesEvidencia"]

        # Obtiene las imagenes de evidencia de la seccion que se va a guardar
        imagenes_evidencias = request.FILES.getlist("imagenes-evidencia[]")

        #return HttpResponse(f"NUM: {num_imagenes_evidencia}")


        #return HttpResponse(len(imagenes_evidencias))

        # Obtiene el registro de la seccion en especifico del reporte
        seccion_reporte_gc = SeccionReporteGardclean.objects.get(seccion_reporte_gardclean_id=seccion_reporte_gc_id)

        if seccion_reporte_gc is None:
            return HttpResponse("No se encontro la seccion",status=HTTPStatus.INTERNAL_SERVER_ERROR)

        # Primero elimina todas las evidencias actuales de la seccion para despues cargar las nuevas
        img_evidencias_rg_actuales = ImagenEvidenciaRg.objects.filter(seccion_reporte_gardclean=seccion_reporte_gc)

        for img_evidencia_rg_actual in img_evidencias_rg_actuales:
            # Elimina el archivo de imagen del servidor
            ruta = f"{STATIC_PREFIX_PATH}/{img_evidencia_rg_actual.ruta}"
            if os.path.isfile(ruta):
                os.remove(ruta)

            img_evidencia_rg_actual.delete()

        # Sube las imagenes al servidor y crea los registros de cada imagen asociados a la seccion con su descripcion correspondiente
        for i in range(0, num_imagenes_evidencia):
            #return HttpResponse(f"NUM: {num_imagenes_evidencia}")
            # Si sobrepasa el numero de imagenes, saldar del bucle
            if len(imagenes_evidencias) == i:
                break
            descripcion = descripciones_imagenes_evidencia[i]
            file = imagenes_evidencias[i]

            # Sube imagen al servidor
            ruta = upload_image_to_server(
                folder_name="img-evidencias-reportes-gc",
                image_name="img-evidencia-rg",
                image_file=file
            )

            # Crea registro en base de datos
            imagen_evidencia_rg = ImagenEvidenciaRg()
            imagen_evidencia_rg.ruta = ruta
            imagen_evidencia_rg.descripcion = descripcion
            imagen_evidencia_rg.seccion_reporte_gardclean = seccion_reporte_gc
            imagen_evidencia_rg.save()

        response["success"] = True

        return JsonResponse(response)
    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)






