from django.http import JsonResponse, HttpResponse
from django.shortcuts import render
# JSON
import json

# Modelos de django
from django.db.models import Q

from oficinas.models import OficinasGestores
# Modelo Usuario ERP
from usuarios.models import Usuario
from clientes.models import CuentaCliente

# Modelo codigo verificacion
from .models import CodigoVerificacion

# Datetime
from datetime import datetime, date, timedelta

# BCrypt
from utils.bcrypt_utils import Bcrypt

# Error codes
from utils.error_codes import INCORRECT_PASSWORD,USER_DOES_NOT_EXISTS, INCORRECT_CODE, EXPIRED_CODE

# Http Status
from http import HTTPStatus

# Singleton Email
from utils.email_utils import Email

# csrf
from django.views.decorators.csrf import csrf_exempt

# Roles de usuario
from utils.user_roles import RH_RANGO, ADMIN_RANGO, CABINA_RANGO, CONTRALOR_RANGO, COORDINADOR_RANGO, COMPRAS_RANGO, SUPERVISOR_RANGO

# Account utils
from backendauth.utils.account_utils import verify_email_account_exists, which_account_type

# Account types
from utils.account_types import AccountType

# Static URLS
from utils.static_constants import STATIC_FULL_URL_PATH

# Python
import random

# Vistas de autenticación

def test(request):
    return HttpResponse("OK")

@csrf_exempt
def login(request):
    if request.method == 'POST':
        # Obtiene el cuerpo de inicio de sesion
        login_body = json.loads(request.body)

        # Respuesta de peticion
        response = {
            "error_response": {
                "code": "",
                "message": ""
            },
            "success_response": {
                "email": "",
                "username": "",
                "erp_user_role": None,
                "client_role": None,
                "position": "",
                "client_logo": "",
                "client_name": "",
                "subsidiary_name": "",
                "accountType": "",
                "erp_user_offices": []
            },
            "success": False,
            "error": False,
            "code": -1
        }

        # Obtiene los datos individuales del cuerpo de login
        email = login_body["email"]
        password = login_body["password"]

        # Verifica que tipo de cuenta va a iniciar sesión
        # - Cuenta de cliente
        # - Usuario ERP (Administrador y Supervisor)
        # - Cuenta de Felipe Neri (coordinador_proyectos@gardclean.com)
        correo_felipe_neri = "coordinador_proyectos@gardclean.com"
        # Nota: Solo esos tipos de rango se permite iniciar sesión
        usuario_erp_email_query = Q(correo=email)
        usuario_erp_rol_query = Q(rango_permiso=ADMIN_RANGO)
        usuario_erp_rol_query.add(Q(rango_permiso=SUPERVISOR_RANGO), Q.OR)
        usuario_erp_rol_query.add(Q(rango_permiso=COORDINADOR_RANGO), Q.OR)
        usuario_erp_rol_query.add(Q(correo=correo_felipe_neri),Q.OR)

        usuario_erp = Usuario.objects.filter(usuario_erp_email_query & usuario_erp_rol_query).first()

        # Si existe un usuario con ese e-mail, supondra que quiere iniciar sesion con una cuenta de usuario ERP
        if usuario_erp is not None:
            if Bcrypt.verify(password, usuario_erp.contrasenia):
                # Obtiene las oficinas del usuario de ERP
                oficinas_gestores = OficinasGestores.objects.filter(ofi_ges_gestor=usuario_erp)

                response["success"] = True
                response["code"] = HTTPStatus.OK.value
                response_dict = response["success_response"]
                response_dict["email"] = email
                response_dict["username"] = usuario_erp.nombre_usuario
                response_dict["erp_user_role"] = usuario_erp.rango_permiso
                response_dict["position"] = usuario_erp.puesto
                response_dict["accountType"] = AccountType.ERP.value
                #response_dict["erp_user_office_id"] = usuario_erp.oficina
                response_dict["erp_user_offices"] = [
                    {
                      "id":  oficina_gestor.ofi_ges_oficina.oficina_id,
                      "nombre": oficina_gestor.ofi_ges_oficina.oficina_nombre
                    }
                    for oficina_gestor in oficinas_gestores
                ]

                response["success_response"] = response_dict

            else:
                response["error"] = True
                response["code"] = HTTPStatus.INTERNAL_SERVER_ERROR.value
                response["error_response"]["code"] = INCORRECT_PASSWORD
                response["error_response"]["message"] = "La contraseña es incorrecta."

        # Verifica si existe un cuenta de cliente con ese email, de ser asi, intentara iniciar sesion con esa cuenta
        cuenta_cliente = CuentaCliente.objects.filter(correo=email).first()

        if cuenta_cliente is not None and usuario_erp is None:
            if Bcrypt.verify(password, cuenta_cliente.contrasenia):
                response["success"] = True
                response["code"] = HTTPStatus.OK.value
                response_dict = response["success_response"]
                response_dict["email"] = email
                response_dict["client_role"] = cuenta_cliente.rol_cliente.rol_cliente_id
                response_dict["accountType"] = AccountType.CLIENTE.value

                response_dict["client_logo"] = f"{STATIC_FULL_URL_PATH}/{cuenta_cliente.cliente.cli_logo_empresa}"
                response_dict["client_name"] = cuenta_cliente.cliente.cli_nombre_cliente
                response_dict["subsidiary_name"] = cuenta_cliente.sucursal.surc_nombre

                response["success_response"] = response_dict
            else:
                response["error"] = True
                response["code"] = HTTPStatus.INTERNAL_SERVER_ERROR.value
                response["error_response"]["code"] = INCORRECT_PASSWORD
                response["error_response"]["message"] = "La contraseña es incorrecta."

        if cuenta_cliente is None and usuario_erp is None:
            response["error"] = True
            response["code"] = HTTPStatus.INTERNAL_SERVER_ERROR.value
            response["error_response"]["code"] = USER_DOES_NOT_EXISTS
            response["error_response"]["message"] = "El correo electrónico ingresado no esta asociado a una cuenta existente."

        return JsonResponse(
            response
        )

    return HTTPStatus(HTTPStatus.METHOD_NOT_ALLOWED)

# Vista que verifica mediante el e-mail si una cuenta de ERP o de cliente existe
@csrf_exempt
def verify_account_exists_view(request):
    if request.method == "GET":
        email = request.GET.get("email")

        # Verifica si existe una cuenta de ERP con el correo
        usuario_erp_email_query = Q(correo=email)
        usuario_erp_rol_query = Q(rango_permiso=ADMIN_RANGO)
        usuario_erp_rol_query.add(Q(rango_permiso=SUPERVISOR_RANGO), Q.OR)
        usuario_erp_rol_query.add(Q(rango_permiso=COORDINADOR_RANGO), Q.OR)

        usuario_erp = Usuario.objects.filter(usuario_erp_email_query & usuario_erp_rol_query).first()

        # Verifica si existe una cuenta de cliente con el correo
        cuenta_cliente = CuentaCliente.objects.filter(correo=email).first()

        response = {
            "exists": (cuenta_cliente is not None or usuario_erp is not None)
        }

        return JsonResponse(response)

    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)


@csrf_exempt
def send_forgot_password_code(request):
    if request.method == 'POST':
        send_code_body = json.loads(request.body)

        # Respuesta de petición
        response = {
            "success": False,
            "error": False,
            "error_message": ""
        }

        # Obtiene el email del cuerpo de la petición
        email = send_code_body["email"]

        # Verifica que la cuenta exista
        if not verify_email_account_exists(email):
            return HttpResponse("El email recibido no esta asociado a ninguna cuenta.", status=HTTPStatus.INTERNAL_SERVER_ERROR.value, )

        # Si existe la cuenta, crea el registro del codigo y envia el e-mail
        codigo = random.randint(100000, 999999)

        # Verifica si actualmente ya existe un registro de un codigo de verificacion
        # con el e-mail del usuario, si no existe, insertara el registro
        if CodigoVerificacion.objects.filter(email_usuario=email).first() is None:
            # Aqui insertara el registro de el codigo de verificacion en la tabla 'codigos_verificacion' de la base de datos
            codigo_verificacion = CodigoVerificacion()
            codigo_verificacion.codigo = str(codigo)
            codigo_verificacion.email_usuario = email
            # Asignara una fecha de expiracion de 15 minutos
            fecha_expiracion = datetime.now() + timedelta(minutes=15)
            codigo_verificacion.fecha_hora_expiracion = fecha_expiracion
            codigo_verificacion.save()
        else:
            # Si ya existe, solamente modificara la fecha de caducidad y el
            # codigo como tal
            codigo_verificacion = CodigoVerificacion.objects.get(email_usuario=email)
            codigo_verificacion.codigo = str(codigo)
            fecha_expiracion = datetime.now() + timedelta(minutes=15)
            codigo_verificacion.fecha_hora_expiracion = fecha_expiracion
            codigo_verificacion.save()

        # Si envia el email correctamente, regresara un estatus de exito
        try:
            Email.send_reset_password_code(destinatario=email, codigo=str(codigo))
            response["success"] = True
            #response["error"] = True
            #response["error_message"] = f"Ocurrio un error al enviar el e-mail: {e}"
            return JsonResponse(response)
        except Exception as e:
            response["error"] = True
            response["error_message"] = f"Ocurrio un error al enviar el e-mail: {e}"
            return JsonResponse(response)
            #return HttpResponse(e,status=HTTPStatus.INTERNAL_SERVER_ERROR.value)


    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

@csrf_exempt
def validate_verification_code(request):
    if request.method == 'POST':
        # Obtiene el cuerpo de la peticion POST
        body = json.loads(request.body)

        # Obtiene los datos individuales del cuerpo de la petición
        email_usuario = body["email"]
        code_user = body["code"]

        # Obtiene el registro(objeto) de el codigo de verificacion enviado(mediante el e-mail del usuario)
        codigo_verificacion_obj = CodigoVerificacion.objects.get(email_usuario=email_usuario)

        # Respuesta
        response = {
            "error": False,
            "error_code": "",
            "success": False
        }

        # Antes de verificar el codigo, primero verifica que el código enviado por e-mail no haya caducado,
        # de ser asi, regresara un error de respuesta
        fecha_caducidad = codigo_verificacion_obj.fecha_hora_expiracion
        if (datetime.now() > fecha_caducidad):
            # El codigo ha caducado
            response["error"] = True
            response["error_code"] = EXPIRED_CODE

            return JsonResponse(response)
        else:
            # Si el codigo no ha caducado, verificara que el codigo pasado por parametro
            # sea el correcto
            if code_user == codigo_verificacion_obj.codigo:
                # El codigo es valido
                codigo_verificacion_obj.delete()
                response["success"] = True
                return JsonResponse(response)
            else:
                # El codigo es incorrecto
                response["error"] = True
                response["error_code"] = INCORRECT_CODE
                return JsonResponse(response)


    else:
        return HttpResponse(status=401)

@csrf_exempt
def change_account_password(request):
    if request.method == 'POST':
        # Respuesta de la petición
        response = {
            "success": False,
            "error": False,
            "error_message": ""
        }

        # Obtiene el cuerpo de la petición POST
        change_password_body = json.loads(request.body)

        # Obtiene los datos individuales del cuerpo de la petición
        email = change_password_body["email"]
        new_password = change_password_body["new_password"]

        # Verifica que la cuenta exista
        if not verify_email_account_exists(email):
            return HttpResponse("El email recibido no esta asociado a ninguna cuenta.", status=HTTPStatus.INTERNAL_SERVER_ERROR.value, )

        # Obtiene el objeto que representa a la cuenta (usuario ERP o cliente)
        cuenta = which_account_type(email)

        # Encripta la nueva contraseña
        new_password = Bcrypt.encrypt(new_password)

        #if account_type == AccountType.ERP:
        if isinstance(cuenta, Usuario):
            try:
                #usuario_erp = Usuario.objects.get(correo=email)
                #usuario_erp.contrasenia = new_password
                #usuario_erp.save()
                cuenta.contrasenia = new_password
                cuenta.save()
                response["success"] = True
            except Exception as e:
                response["error"] = True
                response["error_message"] = e.__str__()

        #if account_type == AccountType.CLIENTE:
        if isinstance(cuenta, CuentaCliente):
            try:
                #cuenta_cliente = CuentaCliente.objects.get(correo=email)
                #cuenta_cliente.contrasenia = new_password
                #cuenta_cliente.save()
                cuenta.contrasenia = new_password
                cuenta.save()
                response["success"] = True
            except Exception as e:
                response["error"] = True
                response["error_message"] = e.__str__()

        return JsonResponse(response)


    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)

# Vista que retornara ciertos datos actualizados de una cuenta de cliente
def get_client_account_data(request):
    if request.method == "GET":
        # Respuesta de la petición
        response = {
            "client_name": "",
            "client_logo": "",
            "subsidiary_name": "",
        }

        # Obtiene los parametros de la peticion
        email = request.GET.get("email")

        # Verifica que la cuenta exista
        if not verify_email_account_exists(email):
            return HttpResponse("El email recibido no esta asociado a ninguna cuenta.", status=HTTPStatus.INTERNAL_SERVER_ERROR.value, )

        # Obtiene el tipo de cuenta
        cuenta = which_account_type(email)

        if isinstance(cuenta, CuentaCliente):
            response["client_name"] = cuenta.cliente.cli_nombre_cliente
            response["client_logo"] = f"{STATIC_FULL_URL_PATH}/{cuenta.cliente.cli_logo_empresa}"
            response["subsidiary_name"] = cuenta.sucursal.surc_nombre

            return JsonResponse(response)

        else:
            return HttpResponse("El email recibido no pertenece a una cuenta de cliente.")


    return HttpResponse(HTTPStatus.METHOD_NOT_ALLOWED)